import { useState } from 'react'
import { useNavigate } from 'react-router-dom'
import { Calendar, CheckCircle, Camera, GraduationCap, Sparkles, Zap, Award, TrendingUp, Users, Clock, Star, BookOpen, User } from 'lucide-react'
import { motion } from 'framer-motion'
import './Enroll.css'

const Enroll = () => {
  const navigate = useNavigate()
  const [selectedLevel, setSelectedLevel] = useState('')
  const [startDate, setStartDate] = useState('')
  const [isSubmitted, setIsSubmitted] = useState(false)

  const courseLevels = [
    { 
      id: 'beginner', 
      name: 'Начинающий', 
      icon: GraduationCap,
      color: '#8B5CF6',
      description: 'Основы фотографии для новичков',
      gradient: 'linear-gradient(135deg, #8B5CF6 0%, #EC4899 100%)',
      duration: '8 недель',
      lessons: '24 урока'
    },
    { 
      id: 'advanced', 
      name: 'Продвинутый', 
      icon: Sparkles,
      color: '#EC4899',
      description: 'Углубленное изучение техник',
      gradient: 'linear-gradient(135deg, #EC4899 0%, #8B5CF6 100%)',
      duration: '12 недель',
      lessons: '36 уроков'
    },
    { 
      id: 'professional', 
      name: 'Профессиональный', 
      icon: Award,
      color: '#F59E0B',
      description: 'Мастерство для профессионалов',
      gradient: 'linear-gradient(135deg, #F59E0B 0%, #EC4899 100%)',
      duration: '16 недель',
      lessons: '48 уроков'
    },
    { 
      id: 'retouch', 
      name: 'Ретушь', 
      icon: Camera,
      color: '#8B5CF6',
      description: 'Обработка и ретушь фотографий',
      gradient: 'linear-gradient(135deg, #8B5CF6 0%, #F59E0B 100%)',
      duration: '6 недель',
      lessons: '18 уроков'
    }
  ]

  const instructors = [
    { name: 'Анна Петрова', experience: '12 лет', rating: 5, specialty: 'Портретная съемка', students: '5000+' },
    { name: 'Михаил Соколов', experience: '15 лет', rating: 5, specialty: 'Пейзажная фотография', students: '7000+' },
    { name: 'Елена Волкова', experience: '10 лет', rating: 5, specialty: 'Студийная съемка', students: '4000+' },
    { name: 'Дмитрий Иванов', experience: '18 лет', rating: 5, specialty: 'Ретушь и обработка', students: '9000+' }
  ]

  const benefits = [
    { icon: Award, title: 'Сертификат', text: 'Официальный сертификат по окончании' },
    { icon: Clock, title: 'Гибкий график', text: 'Учитесь в удобное время' },
    { icon: Users, title: 'Сообщество', text: 'Общение с единомышленниками' },
    { icon: TrendingUp, title: 'Карьерный рост', text: 'Помощь в развитии карьеры' }
  ]

  const handleSubmit = (e) => {
    e.preventDefault()
    if (selectedLevel && startDate) {
      const savedCourses = localStorage.getItem('photoCourses')
      const courses = savedCourses ? JSON.parse(savedCourses) : []
      
      const levelData = courseLevels.find(l => l.id === selectedLevel)
      const randomInstructor = instructors[Math.floor(Math.random() * instructors.length)]
      
      const newCourse = {
        id: Date.now(),
        module: levelData.name,
        instructor: randomInstructor.name,
        completionDate: new Date(new Date(startDate).getTime() + parseInt(levelData.duration) * 7 * 24 * 60 * 60 * 1000).toISOString().split('T')[0],
        rating: 0
      }

      courses.push(newCourse)
      localStorage.setItem('photoCourses', JSON.stringify(courses))
      
      setIsSubmitted(true)
      setTimeout(() => {
        setIsSubmitted(false)
        setSelectedLevel('')
        setStartDate('')
        navigate('/')
      }, 2000)
    }
  }

  const minDate = new Date().toISOString().split('T')[0]
  const maxDate = new Date(Date.now() + 90 * 24 * 60 * 60 * 1000).toISOString().split('T')[0]

  return (
    <div className="enroll-page">
      <div className="container">
        <motion.section 
          className="enroll-hero"
          initial={{ opacity: 0, y: 30 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6 }}
        >
          <motion.div 
            className="enroll-icon-wrapper"
            initial={{ scale: 0, rotate: 180 }}
            animate={{ scale: 1, rotate: 0 }}
            transition={{ type: "spring", stiffness: 200, damping: 15, delay: 0.3 }}
          >
            <Camera size={96} strokeWidth={2.5} className="enroll-hero-icon" />
          </motion.div>
          <h1 className="enroll-title">Записаться на курс</h1>
          <p className="enroll-subtitle">
            Выберите уровень обучения и дату старта для вашего пути в фотографии
          </p>
        </motion.section>

        <motion.section 
          className="enrollment-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.2 }}
        >
          <div className="enrollment-form-wrapper">
            <form className="enrollment-form" onSubmit={handleSubmit}>
              <div className="form-group">
                <label className="form-label">
                  <Camera size={32} strokeWidth={2.5} className="label-icon" />
                  Выберите уровень курса
                </label>
                <div className="levels-grid">
                  {courseLevels.map((level) => {
                    const IconComponent = level.icon
                    return (
                      <motion.button
                        key={level.id}
                        type="button"
                        className={`level-card ${selectedLevel === level.id ? 'selected' : ''}`}
                        onClick={() => setSelectedLevel(level.id)}
                        whileHover={{ y: -4, transition: { duration: 0.2 } }}
                        whileTap={{ scale: 0.98 }}
                        style={{ 
                          borderColor: selectedLevel === level.id ? level.color : 'rgba(139, 92, 246, 0.4)',
                          background: selectedLevel === level.id ? (level.color + '20') : 'rgba(15, 15, 35, 0.95)'
                        }}
                      >
                        <div className="level-icon-wrapper" style={{ background: level.color + '30' }}>
                          <IconComponent size={52} strokeWidth={2.5} color={level.color} />
                        </div>
                        <div className="level-content">
                          <h3 className="level-name">{level.name}</h3>
                          <p className="level-description">{level.description}</p>
                          <div className="level-meta">
                            <span className="level-duration">
                              <Clock size={18} strokeWidth={2} />
                              {level.duration}
                            </span>
                            <span className="level-lessons">
                              <BookOpen size={18} strokeWidth={2} />
                              {level.lessons}
                            </span>
                          </div>
                        </div>
                        {selectedLevel === level.id && (
                          <motion.div 
                            className="level-check"
                            initial={{ scale: 0 }}
                            animate={{ scale: 1 }}
                            style={{ background: level.gradient }}
                          >
                            <CheckCircle size={32} strokeWidth={2.5} color="white" />
                          </motion.div>
                        )}
                      </motion.button>
                    )
                  })}
                </div>
              </div>

              <div className="form-group">
                <label htmlFor="startDate" className="form-label">
                  <Calendar size={32} strokeWidth={2.5} className="label-icon" />
                  Дата старта курса
                </label>
                <input
                  type="date"
                  id="startDate"
                  className="form-input"
                  value={startDate}
                  onChange={(e) => setStartDate(e.target.value)}
                  min={minDate}
                  max={maxDate}
                  required
                />
              </div>

              <motion.button 
                type="submit" 
                className="submit-button" 
                disabled={!selectedLevel || !startDate}
                whileHover={{ scale: 1.03, y: -4 }}
                whileTap={{ scale: 0.98 }}
              >
                <span>Записаться на курс</span>
                <CheckCircle size={32} strokeWidth={2.5} className="btn-icon" />
              </motion.button>

              {isSubmitted && (
                <motion.div 
                  className="success-message"
                  initial={{ opacity: 0, scale: 0.9 }}
                  animate={{ opacity: 1, scale: 1 }}
                  transition={{ duration: 0.3 }}
                >
                  <CheckCircle size={40} strokeWidth={2.5} className="success-icon" />
                  <span>Вы успешно записались на курс!</span>
                </motion.div>
              )}
            </form>
          </div>
        </motion.section>

        <motion.section 
          className="benefits-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.4 }}
        >
          <h2 className="section-title">Преимущества обучения</h2>
          <div className="benefits-grid">
            {benefits.map((benefit, index) => {
              const IconComponent = benefit.icon
              return (
                <motion.div
                  key={benefit.title}
                  className="benefit-card"
                  initial={{ opacity: 0, y: 20 }}
                  animate={{ opacity: 1, y: 0 }}
                  transition={{ delay: index * 0.1, duration: 0.5 }}
                  whileHover={{ y: -4, transition: { duration: 0.2 } }}
                >
                  <div className="benefit-icon">
                    <IconComponent size={44} strokeWidth={2.5} color="#8B5CF6" />
                  </div>
                  <h3 className="benefit-title">{benefit.title}</h3>
                  <p className="benefit-text">{benefit.text}</p>
                </motion.div>
              )
            })}
          </div>
        </motion.section>

        <motion.section 
          className="instructors-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.5 }}
        >
          <h2 className="section-title">Наши преподаватели</h2>
          <div className="instructors-grid">
            {instructors.map((instructor, index) => (
              <motion.div
                key={instructor.name}
                className="instructor-card"
                initial={{ opacity: 0, scale: 0.9 }}
                animate={{ opacity: 1, scale: 1 }}
                transition={{ delay: index * 0.1, duration: 0.5 }}
                whileHover={{ y: -4, transition: { duration: 0.2 } }}
              >
                <div className="instructor-avatar">
                  <User size={44} strokeWidth={2.5} />
                </div>
                <h3 className="instructor-name">{instructor.name}</h3>
                <p className="instructor-experience">{instructor.experience} опыта</p>
                <div className="instructor-rating">
                  {Array.from({ length: 5 }, (_, i) => (
                    <Star key={i} size={18} fill="#F59E0B" stroke="#F59E0B" strokeWidth={2} />
                  ))}
                </div>
                <p className="instructor-specialty">{instructor.specialty}</p>
                <div className="instructor-students">
                  <Users size={18} strokeWidth={2} />
                  <span>{instructor.students} студентов</span>
                </div>
              </motion.div>
            ))}
          </div>
        </motion.section>
      </div>
    </div>
  )
}

export default Enroll

